# Clare Andrews juvenile starling experiment, data analysis script
# Daniel Nettle, February 2021 on the basis of earlier scripts by Clare Andrews and Melissa Bateson
# Modified version of August 2021

###Head#####
library(tidyverse)
library(psych)
library(lmerTest)
library(effsize)
library(cowplot)
library(emmeans)
library(metafor)
library(rptR)
library(metap)
theme_set(theme_classic())

# Get data
# Food consumption
load("Consumption.Rdata")
# Weight data
load("Mass.Rdata")
# Telomere data 
load("telomeres.Rdata")

# Define necessary variables and do some renaming
d$Insecurity="Control"
d$Insecurity[d$Treatment=="Insecure" & d$TreatmentWeek>0]="Insecure"
d$Insecurity=as.factor(d$Insecurity)
d$TreatmentWeekfactor=as.factor(d$TreatmentWeek)
contrasts(d$Insecurity)=contr.sum(2)
contrasts(d$TreatmentWeekfactor)=contr.sum(9)
t$Insecurity="Control"
t$Insecurity[t$Treatment=="Insecure" & t$TreatmentWeek>0]="Insecure"
t$Insecurity=as.factor(t$Insecurity)
t$TreatmentWeekfactor=as.factor(t$TreatmentWeek)
contrasts(t$Insecurity)=contr.sum(2)
contrasts(t$TreatmentWeekfactor)=contr.sum(5)
f$Insecurity="Control"
f$Insecurity[f$Treatment=="Insecure" & f$TreatmentWeek>0]="Insecure"
f$Insecurity=as.factor(f$Insecurity)
f$TreatmentWeekfactor=as.factor(f$TreatmentWeek)
f$Insecurity=as.factor(f$Insecurity)
f$TreatmentWeekfactor=as.factor(f$TreatmentWeek)
contrasts(f$Insecurity)=contr.sum(2)
contrasts(f$TreatmentWeekfactor)=contr.sum(19)
st=subset(t, TreatmentWeek>0)
##End head

####Mass analysis#####
# Sex difference in mass
s1=lmer(Mass~SexVisual + TreatmentWeekfactor  + (1|BirdID), data=subset(d, TreatmentWeek>=0))
summary(s1)
anova(s1)
emmeans(s1, specs=~SexVisual)

# Tarsus length
ICC(data.frame(d$Tars1[d$Week==0], d$Tars2[d$Week==0]))

# Is there a difference is mass by treatment at baseline?
t.test(d$Mass[d$TreatmentWeek==0]~d$Treatment[d$TreatmentWeek==0])
describeBy(d$Mass[d$TreatmentWeek==0], d$Treatment[d$TreatmentWeek==0])
summary(lm(Mass~Treatment + AvTars, data=subset(d, TreatmentWeek==0)))
anova(lm(Mass~Treatment + AvTars, data=subset(d, TreatmentWeek==0)))

# Modelling mass by insecurity
m1=lmer(Mass~Insecurity*TreatmentWeekfactor + AvTars + (1|BirdID), data=subset(d, TreatmentWeek>=0))
summary(m1)
anova(m1)
# Does including sex in the model make any difference?
m1s=lmer(Mass~Insecurity*TreatmentWeekfactor*SexVisual + AvTars  + (1|BirdID), data=subset(d, TreatmentWeek>=0))
summary(m1s)
anova(m1s)

# Calculate the effect size at different points
describeBy(subset(d, TreatmentWeek==5)$Mass, subset(d, TreatmentWeek==8)$Treatment)
describeBy(subset(d, TreatmentWeek==8)$Mass, subset(d, TreatmentWeek==8)$Treatment)
describeBy(subset(d, TreatmentWeek==14)$Mass, subset(d, TreatmentWeek==14)$Treatment)
describeBy(subset(d, TreatmentWeek==19)$Mass, subset(d, TreatmentWeek==19)$Treatment)
effsize::cohen.d(Mass~Treatment, data=subset(d, TreatmentWeek==5))
effsize::cohen.d(Mass~Treatment, data=subset(d, TreatmentWeek==8))
effsize::cohen.d(Mass~Treatment, data=subset(d, TreatmentWeek==19))
effsize::cohen.d(Mass~Treatment, data=subset(d, TreatmentWeek==14))

# Figure 1A
massfigdata=subset(d, TreatmentWeek>=0) %>% group_by(TreatmentWeek, Treatment) %>% 
  summarise(meanmasschange=mean(MassChange), se=describe(MassChange)$se)
massfigdata$dummy=c(1, 1, 2, 2, 3, 3, 4, 4, 5, 5, 6, 6, 7, 7, 8, 8)
fig1A=ggplot(massfigdata, aes(x=as.factor(TreatmentWeek), y=meanmasschange, colour=Treatment)) + 
  geom_point(position=position_dodge(0.1)) + 
  geom_line(aes(x=dummy)) + 
  geom_errorbar(aes(ymax=meanmasschange+se, ymin=meanmasschange-se), width=0.2, position=position_dodge(0.1)) + 
  geom_line(aes()) + 
  ylab("Mass change from baseline (g)") + 
  xlab("Weeks of treatment") + 
  scale_colour_manual(values=c("darkblue", "darkred")) + 
  geom_vline(xintercept=1, linetype="dotted") + 
  theme(legend.position=c(0.5, 0.25), legend.title=element_blank())
fig1A

# Fat scoring
# Correlation with mass
cor.test(d$FatScoreCA, d$Mass, use="complete.obs")
# Inter-rater reliability
sum(xtabs(~d$FatScoreCA))
sum(xtabs(~d$FatScoreLH))
ICC(data.frame(d$FatScoreCA, d$FatScoreLH))
cor.test(d$FatScoreCA, d$FatScoreLH, use="complete.obs")
describe(d$FatScoreCA)
t.test(d$FatScoreCA~d$Treatment)
describeBy(d$FatScoreCA, d$Treatment)
effsize::cohen.d(FatScoreCA~Treatment, data=d)
# Including sex
summary(lm(FatScoreCA~Treatment*SexVisual, data=d))

# Figure 1B
fig1B=ggplot(d, aes(x=Treatment, y=FatScoreCA)) + 
  geom_boxplot(width=0.5) + 
  geom_dotplot(binaxis = "y", stackdir = "center", dotsize=0.5, aes(fill=Treatment, colour=Treatment)) + 
  ylab("Fat score") + 
  scale_fill_manual(values=c("darkblue", "darkred"))  +
  scale_colour_manual(values=c("darkblue", "darkred"))  +
  guides(fill=FALSE, colour=FALSE)
fig1B

###Food consumption analysis#####
# Averaging by week
fw = f %>% group_by(TreatmentWeek, Insecurity, Flight) %>% summarise(Consumption=mean(ConsumedPerBird, na.rm=T))
fw2=subset(fw, TreatmentWeek>0)
# Descriptives
describeBy(fw2$Consumption, fw2$Insecurity)
effsize::cohen.d(Consumption~Insecurity, data=fw2)

# Model
c1=lmer(Consumption~Insecurity*as.factor(TreatmentWeek) + (1|Flight), data=fw2)
summary(c1)
anova(c1)
emmeans(c1, specs=~Insecurity)
# Figure for consumption
fig1C=ggplot(fw2, aes(x=Insecurity, y=Consumption)) + 
  geom_boxplot(outlier.shape=NA, colour="black", width=0.5) +
  geom_dotplot(binaxis = "y", stackdir = "center", dotsize=0.5, aes(fill=Insecurity, colour=Insecurity)) + 
  ylab("Food consumed per bird (g)") + 
  xlab("Treatment") + 
  scale_colour_manual(values=c("darkblue", "darkred")) + 
  scale_fill_manual(values=c("darkblue", "darkred")) + 
  guides(fill=FALSE, colour=FALSE)
fig1C


# Assemble figure 1
png("figure1.png", res=300, width=9*300, height=3*300)
plot_grid(fig1A, fig1B, fig1C, labels=c("A", "B", "C"), nrow=1, rel_widths=c(0.5, 0.25, 0.25))
dev.off()

###Telomeres######
# ICC
rpt(AverageTL ~ Week + (1|BirdID), grname="BirdID", data=t, datatype="Gaussian", nboot=1000, npermut=0)

# Descriptives
output = subset(t, TreatmentWeek==-2) %>% group_by(BirdID) %>% summarise(birdav = mean(AverageTL, na.rm=T))
describe(output$birdav)

# Consistency of individual TL over time
# Restructure data for correlation matrix
wt = st %>% group_by(BirdID, TreatmentWeek) %>% summarise(AvTL=mean(AverageTL, na.rm=T)) %>%
    pivot_wider(values_from = AvTL, names_from=TreatmentWeek)
colnames(wt)=c("BirdID", "AvTL2", "AvTL8", "AvTL14", "AvT19")
wt=merge(wt, select(st, BirdID, BaselineAvTL), by="BirdID", all.x=T, all.y=F)
cor(wt[ , 2:6], use="pairwise.complete.obs")

# Telomere shortening over the course of the study
#Overall
st %>% 
  subset(TreatmentWeek==19) %>% 
  group_by(BirdID, Treatment) %>% 
  summarise(iaTL = mean(AverageTL, na.rm=T), iBaselineAvTL = mean(BaselineAvTL, na.rm=T)) %>%
  mutate(deltaTL = iaTL - iBaselineAvTL) %>%
  ungroup() %>%  
  summarise(meanDelta=mean(deltaTL, na.rm=T), sdDelta=sd(deltaTL, na.rm=T), N=n(), seDelta=describe(deltaTL)$se)

sst = st %>% 
  subset(TreatmentWeek==19) %>% 
  group_by(BirdID, Treatment) %>% 
  summarise(iaTL = mean(AverageTL, na.rm=T), iBaselineAvTL = mean(BaselineAvTL, na.rm=T)) %>%
  mutate(deltaTL = iaTL - iBaselineAvTL) %>%
  ungroup() 
t.test(sst$deltaTL, mu=0)

# By treatment  
st %>% 
  subset(TreatmentWeek==19) %>% 
  group_by(BirdID, Treatment) %>% 
  summarise(iaTL = mean(AverageTL, na.rm=T), iBaselineAvTL = mean(BaselineAvTL, na.rm=T)) %>%
  mutate(deltaTL = iaTL - iBaselineAvTL) %>%
  group_by(Treatment) %>%
  summarise(meanDelta=mean(deltaTL, na.rm=T), sdDelta=sd(deltaTL, na.rm=T), N=n(), seDelta=describe(deltaTL)$se)

t.test(subset(sst, Treatment=="Insecure")$deltaTL, mu=0)
t.test(subset(sst, Treatment=="Control")$deltaTL, mu=0)

# Telomere length at baseline by treatment
dat = st %>% subset(TreatmentWeek==2) %>% group_by(BirdID, Treatment) %>% summarise(aTL=mean(BaselineAvTL, na.rm=T))
describeBy(dat$aTL, dat$Treatment)
t.test(dat$aTL ~ dat$Treatment)

# Simple model of treatment effect on aTL
t1 =lmer(AverageTL ~TreatmentWeekfactor*Insecurity + (1|BirdID), data=t)
summary(t1)
anova(t1)
effsize::cohen.d(AverageTL~Treatment, data=subset(t, TreatmentWeek==2))
effsize::cohen.d(AverageTL~Treatment, data=subset(t, TreatmentWeek==8))
effsize::cohen.d(AverageTL~Treatment, data=subset(t, TreatmentWeek==14))
effsize::cohen.d(AverageTL~Treatment, data=subset(t, TreatmentWeek==19))
describeBy(subset(t, TreatmentWeek==2)$AverageTL, subset(t, TreatmentWeek==2)$Treatment)
17093-17609
describeBy(subset(t, TreatmentWeek==14)$AverageTL, subset(t, TreatmentWeek==14)$Treatment)
17401-17197

# For parameter estimate, model with non-significant interaction suppressed
t1 =lmer(AverageTL ~TreatmentWeekfactor + Insecurity + (1|BirdID), data=t)
summary(t1)

# Version including sex
t1s =lmer(AverageTL ~TreatmentWeekfactor*Insecurity*SexVisual + (1|BirdID), data=t)
summary(t1s)
anova(t1s)

# Figure 2A
ttt2 = t %>% group_by(BirdID, TreatmentWeekfactor, Treatment) %>% summarise(iTL=mean(AverageTL, na.rm=T)) %>%
  group_by(Treatment, TreatmentWeekfactor) %>%
  summarise(TL=mean(iTL, na.rm=T), seTL=describe(iTL)$se) 
ttt2$TreatmentWeekfactor=relevel(ttt2$TreatmentWeekfactor, ref="-2")

fig2A= ggplot(ttt2, aes(x=as.numeric(as.character(TreatmentWeekfactor)), y=TL, colour=Treatment)) + 
  geom_point() + 
  geom_errorbar(aes(ymax=TL+seTL, ymin=TL-seTL), width=0.5) + 
  geom_line(aes(x=as.numeric(as.character(TreatmentWeekfactor)))) + 
  scale_x_continuous(breaks=c(0, 2, 8, 14, 19)) + 
  ylab("Average TL (bp)") + 
  geom_vline(xintercept=0.1, linetype="dotted") + 
  scale_colour_manual(values=c("darkblue", "darkred")) + 
  xlab("Weeks of treatment") + 
  theme(legend.title=element_blank(), legend.position=c(0.5, 0.9)) + 
  coord_cartesian(ylim=c(16750, 18250))
fig2A

# Figure 2A alternate version
colnames(t)
ttt2 = t %>% group_by(BirdID, TreatmentWeekfactor, Treatment) %>% summarise(iTL=mean(AverageTL, na.rm=T), iBaseline=mean(BaselineAvTL, na.rm=T)) %>%
  mutate(iDeltaTL = iTL - iBaseline) %>%
  group_by(Treatment, TreatmentWeekfactor) %>%
  summarise(TL=mean(iTL, na.rm=T), seTL=describe(iTL)$se, DeltaTL=mean(iDeltaTL, na.rm=T), seDeltaTL=describe(iDeltaTL)$se) 
ttt2$TreatmentWeekfactor=relevel(ttt2$TreatmentWeekfactor, ref="-2")
ttt2$wsDeltaTL = ttt2$DeltaTL + mean(ttt2$TL, na.rm=T)
fig2A= ggplot(ttt2, aes(x=as.numeric(as.character(TreatmentWeekfactor)), y=TL, colour=Treatment)) + 
  geom_point() + 
  geom_errorbar(aes(ymax=TL+seTL, ymin=TL-seTL), width=0.5) + 
  geom_line(aes(x=as.numeric(as.character(TreatmentWeekfactor)))) + 
  scale_x_continuous(breaks=c(0, 2, 8, 14, 19)) + 
  ylab("Average TL (bp)") + 
  geom_vline(xintercept=0.1, linetype="dotted") + 
  scale_colour_manual(values=c("darkblue", "darkred")) + 
  xlab("Weeks of treatment") + 
  theme(legend.title=element_blank(), legend.position=c(0.5, 0.9)) +
  coord_cartesian(ylim=c(16750, 18250))
fig2A

# Percentile analysis of telomeres #
# Make long thin version of data
keeps=c("BirdID", "TubeLabel", "TreatmentWeekfactor", "Insecurity", colnames(t)[13:29], "SexVisual")
tl= t %>% select(keeps) 
tl=pivot_longer(tl, cols= colnames(t)[13:29], names_to="Percentile", values_to="TL")
tl$Percentile=as.factor(tl$Percentile)
tl$Percentile=relevel(tl$Percentile, ref="X0.1")
tl$PercentileNumeric=as.numeric(substr(tl$Percentile, start=2, stop=5))
tl$Percentile=relevel(tl$Percentile, ref="X0.5")
# Fit model
tl$Insecurity=factor(tl$Insecurity)
t2=lmer(TL~Insecurity*Percentile*TreatmentWeekfactor + (1|BirdID/TubeLabel), REML=T, data=tl)
summary(t2)
anova(t2)

# Version including sex (four-way interactions, ugh!)
t2s=lmer(TL~Insecurity*Percentile*TreatmentWeekfactor*SexVisual + (1|BirdID/TubeLabel), REML=T, data=tl)
summary(t2s)
anova(t2s)

# Make figure for telomere percentile
# Generate marginal means
ml2=lmer(TL~Insecurity*Percentile + (1|BirdID/TubeLabel), REML=T, data=tl)
summary(ml2)
anova(ml2)

# Make data frame of marginal means
figdat=as.data.frame(emmeans(ml2, specs=~Insecurity+Percentile))
# Now reshape this to calculate difference scores
figdat2 = select(figdat, Insecurity, Percentile, emmean, SE) %>% pivot_wider(names_from=Insecurity, values_from=c(emmean, SE))
figdat2$delta=figdat2$emmean_Insecure-figdat2$emmean_Control
figdat2$deltaSE=sqrt(figdat2$SE_Control^2 + figdat2$SE_Insecure^2)
figdat2$deltaz=figdat2$delta/figdat2$SE_Control
figdat2$Percentilenumeric=seq(10, 90, by=5)
# Make the figure
fig2B=ggplot(figdat2, aes(x=Percentilenumeric, y=delta)) + 
  geom_point() + 
  geom_errorbar(aes(ymin=delta-deltaSE, ymax=delta+deltaSE), width=2) + 
  theme_classic() + 
  geom_hline(yintercept=0, linetype="dotted") + 
  ylab("Difference in TL (bp)") + 
  xlab("Percentile of TL distribution") + 
  scale_x_continuous(breaks=seq(0, 100, by=10))
fig2B

# Assemble telomere figure 
png("figure2.png", res=300, width=8*300, height=3*300)
plot_grid(fig2A, fig2B, labels=c("A", "B"))
dev.off()

###Feather regrowth######
d$Featheravg=(d$Feather + d$FeatherRpt)/2
describeBy(d$Featheravg, paste(d$TreatmentWeek, d$Treatment))
# In treatment week 17 there are three control birds whose feathers have got much shorter from week 16 - they broke
d$Featheravg[d$TreatmentWeek==17 & d$Treatment=="Control"]
# Replace these with NA
d$Featheravg[which(d$TreatmentWeek==17 & d$Treatment=="Control" & d$Featheravg<30)]=NA

# Statistical model
fd=subset(d, TreatmentWeek>0 & d$TreatmentWeek<21)
f1=lmer(Featheravg ~ as.factor(TreatmentWeek)*Treatment + (1|BirdID), data=fd)
summary(f1)
anova(f1)
# Without interaction for treatment parameter estimate
f1b=lmer(Featheravg ~ as.factor(TreatmentWeek)+Treatment + (1|BirdID), data=fd)
summary(f1b)

# Estimate effect size by average across weeks
fd %>% group_by(Treatment, BirdID) %>%
  summarise(meanf=mean(Featheravg, na.rm=T)) %>%
  group_by(Treatment) %>%
  summarise(mf=mean(meanf, na.rm=T), se=describe(meanf)$se)
# Cohen d
cd=fd %>% group_by(Treatment, BirdID) %>%
  summarise(meanf=mean(Featheravg, na.rm=T)) 
cohen.d(cd$meanf, cd$Treatment)

# Version of model with sex included
f1s=lmer(Featheravg ~ as.factor(TreatmentWeek)*Treatment*SexVisual + (1|BirdID), data=fd)
summary(f1s)
anova(f1s)

# Make a figure
f2=lmer(Featheravg~as.factor(TreatmentWeek)*Treatment + (1|BirdID), data=fd)
fig3dat=as.data.frame(emmeans(f2, specs=~Treatment + TreatmentWeek))
colnames(fig3dat)
extra.rows=data.frame(
  c("Control", "Insecure", "Control", "Insecure", "Control", "Insecure"), 
  c(-2, -2, 5.2, 5.2, 17.2, 17.2), 
  c(0, 0, 0, 0, 0, 0), 
  c(0, 0, 0, 0, 0, 0), 
  c(NA, NA, NA, NA, NA, NA),
  c(NA, NA, NA, NA, NA, NA), 
  c(NA, NA, NA, NA, NA, NA))
colnames(extra.rows)=colnames(fig3dat)
fig3dat=rbind(fig3dat, extra.rows)
fig3dat$Growthweek1=NA
fig3dat$Growthweek2=NA
fig3dat$Growthweek3=NA
fig3dat$Growthweek1[fig3dat$TreatmentWeek<=5]=fig3dat$TreatmentWeek[fig3dat$TreatmentWeek<=5]
fig3dat$Growthweek2[fig3dat$TreatmentWeek>5 & fig3dat$TreatmentWeek<=17]=fig3dat$TreatmentWeek[fig3dat$TreatmentWeek>5 & fig3dat$TreatmentWeek<=17]
fig3dat$Growthweek3[fig3dat$TreatmentWeek>17]=fig3dat$TreatmentWeek[fig3dat$TreatmentWeek>17]

fig3=ggplot(fig3dat, aes(x=TreatmentWeek, y=emmean, colour=Treatment)) + 
  geom_point() + 
  geom_errorbar(aes(ymax=emmean+SE, ymin=emmean-SE), width=0.5) + 
  geom_line(aes(x=Growthweek1)) + 
  geom_line(aes(x=Growthweek2)) + 
  geom_line(aes(x=Growthweek3)) + 
  scale_colour_manual(values=c("darkblue", "darkred"))  + 
  ylab("Feather length (mm)")  + 
  coord_cartesian(xlim=c(-2, 20), ylim=c(0, 70)) + 
  geom_vline(xintercept=-2) + 
  geom_vline(xintercept=5.2) + 
  geom_vline(xintercept=17.2) + 
  geom_vline(xintercept=0, linetype="dotted") + 
  scale_x_continuous(breaks=c(-2, 0, 2, 5, 8, 11, 14, 17, 19)) + 
  xlab("Weeks of treatment") + 
  theme(legend.title=element_blank())
fig3  

png("Figure3.png", res=300, width=6*300, height=3*300)
fig3
dev.off()


###Meta-analysis####

# Energy storage
contrasts(d$Insecurity)=contr.treatment(2)
mm1=lmer(scale(Mass)~Insecurity+TreatmentWeekfactor + AvTars + (1|BirdID), data=subset(d, TreatmentWeek>=0), REML=T)
summary(mm1)
mm2=lm(scale(FatScoreCA)~Insecurity, data=d)
summary(mm2)
allmetap(c(0.098520, 0.00746), method="sumz")
ma1=rma(yi=c(summary(mm1)$coefficients[2, 1], 
             summary(mm2)$coefficients[2, 1]), 
        sei=c(summary(mm1)$coefficients[2, 2], 
          summary(mm2)$coefficients[2, 2]), method="FE")
summary(ma1)
forest(ma1, slab=c("Mass", "Fat score"), psize=1, xlab="Standardized parameter estimate")
# Somatic investment
mm3=lmer(scale(Featheravg) ~ Treatment + as.factor(TreatmentWeek) + (1|BirdID), data=fd)
summary(mm3)
contrasts(t$Insecurity)=contr.treatment(2)
mm4=lmer(scale(AverageTL) ~ Insecurity + TreatmentWeekfactor + (1|BirdID), data=t)
summary(mm4)
allmetap(c(0.0231, 0.0896), method="sumz")

ma2=rma(yi=c(summary(mm3)$coefficients[2, 1], 
             summary(mm4)$coefficients[2, 1]), 
        sei=c(summary(mm3)$coefficients[2, 2], 
              summary(mm4)$coefficients[2, 2]), method="FE")
summary(ma2)
forest(ma2, slab=c("Feather regrowth", "Average TL"), psize=1, xlab="Standardized parameter estimate")

# Combined forest plot (figure 5)
yis=c(summary(mm1)$coefficients[2, 1], 
      summary(mm2)$coefficients[2, 1],
      summary(mm3)$coefficients[2, 1],
      summary(mm4)$coefficients[2, 1])
seis=c(summary(mm1)$coefficients[2, 2], 
       summary(mm2)$coefficients[2, 2],
       summary(mm3)$coefficients[2, 2],
       summary(mm4)$coefficients[2, 2])
png("figure4.png", res=300, width=7*300, height=5*300)
forest.default(x=yis, sei=seis, 
               slab=c("Mass", "Fat score", "Feather regrowth", "Average TL"), 
                psize=1, 
               xlab="Standardized parameter estimate", 
               ylim=c(0, 9),
               rows=c(6, 5, 3, 2), 
               alim=c(-1, 1.5), 
               xlim=c(-3, 2.5), 
               at=c(-1, -0.5, 0, 0.5, 1, 1.5), 
               cex=0.85)
addpoly(ma1, row=4, mlab="Pooled effect for mass and fat score", cex=0.85)
addpoly(ma2, row=1, mlab="Pooled effect for feather regrowth and TL", cex=0.85)
dev.off()